package connectivity;

import com.datecs.fiscalprinter.SDK.FiscalDeviceV1;
import com.datecs.fiscalprinter.SDK.FiscalPrinterV1;
import com.datecs.fiscalprinter.SDK.TransportProtocolV1;
import com.datecs.fiscalprinter.SDK.model.BGR.*;
import com.datecs.fiscalprinter.SDK.model.DatecsFiscalDevice;
import com.datecs.fiscalprinter.SDK.model.UserLayerV1.cmdService;

/**
 * @author DatecsSoft_Team
 */

public interface PrinterConnection {
    DatecsFiscalDevice open(String portName) throws Exception;

    /**
     * createPrinter:
     * <p>
     * Common method to create the instance of a specific fiscal drive,
     * depending on the type of communication protocol and the desired error handling settings.
     * <p>
     * Datecs JavaSDK exceptions:
     * <p>
     * Datecs JavaSDK throws a Status Bytes exception,
     * The Method initCriticalStatusPrinter() - enables users to select Status Bytes of device
     * to be a critical exception for their application or not.
     * For example, myCriticalStatusSet[0][3]-"No client display connected"
     * may be a critical error in a POS application running in a store.
     * And when printer is an office application and no need of this exception  must to turned off.
     * After the execution of each command, the device returns a response or error message in the format:
     *
     * @DescriptionOfError. This option is turned off by default, use the static method:
     * cmdService().setErrorMessageInResponse(true);  to turn it on if necessary.
     * Valid only at DATECS FP-800 / FP-2000 / FP-650 / SK1-21F / SK1-31F/ FMP-10 / FP-550.
     * Use setThrowErrorMessage(true) to include these messages in the exception.
     * The exception messages has the following format: Status Byte + separator + @Description Of Error.
     * @note: Use fiscalDevice.getConnectedPrinterV1().setMsgSeparator(); to setup desired separator, by default
     * separator is "\n\r"
     * <p>
     * All messages are in Bulgarian.
     **/
    static DatecsFiscalDevice createPrinter(String modelName, TransportProtocolV1 transportProtocolV1) throws Exception {
        DatecsFiscalDevice connectedFD = new DatecsFiscalDevice();
        switch (modelName) {
            case "DP-05":
                //Protocol V1 -Identifies all 6 byte status devices !!!
                //Enables whether a status bytes of the device is a critical exception for user application !
                FiscalDeviceV1.setCriticalStatuses(initCriticalStatusECR());
                connectedFD.setConnectedModel(new DP05_BGR(transportProtocolV1));
                return connectedFD;
            case "DP-15":
                FiscalDeviceV1.setCriticalStatuses(initCriticalStatusECR());
                connectedFD.setConnectedModel(new DP15_BGR(transportProtocolV1));
                return connectedFD;
            case "DP-25":
                FiscalDeviceV1.setCriticalStatuses(initCriticalStatusECR());
                connectedFD.setConnectedModel(new DP25_BGR(transportProtocolV1));
                return connectedFD;
            case "DP-35":
                FiscalDeviceV1.setCriticalStatuses(initCriticalStatusECR());
                //Not tested with device
                connectedFD.setConnectedModel(new DP35_BGR(transportProtocolV1));
                return connectedFD;
            case "WP-50":
                FiscalDeviceV1.setCriticalStatuses(initCriticalStatusECR());
                //Not tested with device
                connectedFD.setConnectedModel(new WP50_BGR(transportProtocolV1));
                return connectedFD;
            case "DP-150":
                FiscalDeviceV1.setCriticalStatuses(initCriticalStatusECR());
                connectedFD.setConnectedModel(new DP150_BGR(transportProtocolV1));
                return connectedFD;
            case "FP-700":
                FiscalPrinterV1.setCriticalStatuses(initCriticalStatusPrinter());
                connectedFD.setConnectedModel(new FP700_BGR(transportProtocolV1));
                return connectedFD;
            case "FP-800":
                FiscalPrinterV1.setCriticalStatuses(initCriticalStatusPrinter());
                connectedFD.setConnectedModel(new FP800_BGR(transportProtocolV1));
                return connectedFD;
            case "FP-650":
                FiscalPrinterV1.setCriticalStatuses(initCriticalStatusPrinter());
                //Creation of  Device Instance
                connectedFD.setConnectedModel(new FP650_BGR(transportProtocolV1));
                new cmdService().setErrorMessageInResponse(true);
                return connectedFD;
            case "FMP-10":
                FiscalPrinterV1.setCriticalStatuses(initCriticalStatusPrinter());
                connectedFD.setConnectedModel(new FMP10_BGR(transportProtocolV1));
                new cmdService().setErrorMessageInResponse(true);
                return connectedFD;
            case "FP-2000":
                FiscalPrinterV1.setCriticalStatuses(initCriticalStatusPrinter());
                //Not tested with device
                connectedFD.setConnectedModel(new FP2000_BGR(transportProtocolV1));
                return connectedFD;
            case "SK1-21F ":
                FiscalPrinterV1.setCriticalStatuses(initCriticalStatusPrinter());
                //Not tested
                connectedFD.setConnectedModel(new SK1_21F_BGR(transportProtocolV1));
                return connectedFD;
            case "SK1-31F":
                FiscalPrinterV1.setCriticalStatuses(initCriticalStatusPrinter());
                //Not tested
                connectedFD.setConnectedModel(new SK1_31F_BGR(transportProtocolV1));
                return connectedFD;
            case "FP-550":
                FiscalPrinterV1.setCriticalStatuses(initCriticalStatusPrinter());
                connectedFD.setConnectedModel(new FP550_BGR(transportProtocolV1));
                return connectedFD;
            default:
                throw new Exception("Unsupported model:" + modelName);
        }

    }


    void close() throws Exception;

    void setBaudRate(int baudRate) throws Exception;

    void setTimeout(int timeout) throws Exception;

    void setPortName(String portName) throws Exception;

    String getPortName() throws Exception;

    //  Object getSyncObject() throws Exception;

    boolean isSearchByBaudRateEnabled();


    /**
     * Enables whether a status is a critical exception
     *
     * @return
     */
    static boolean[][] initCriticalStatusPrinter() {

        boolean[][] myCriticalStatusSet = new boolean[6][8];
        myCriticalStatusSet[0][7] = false; //"For internal use  1."
        myCriticalStatusSet[0][6] = true;//"Cover is Open"
        myCriticalStatusSet[0][5] = true; // General error - this is OR of all errors marked with #.
        myCriticalStatusSet[0][4] = true;//"# Failure in printing mechanism"
        myCriticalStatusSet[0][3] = false;//"No client display connected"
        myCriticalStatusSet[0][2] = false;//"The real time clock is not synchronized"
        myCriticalStatusSet[0][1] = true;//"# Command code is invalid"
        myCriticalStatusSet[0][0] = true;//"# Syntax error"
        myCriticalStatusSet[1][7] = false;//"For internal use  1"
        myCriticalStatusSet[1][6] = true;//"The built-in tax terminal does not responding"
        myCriticalStatusSet[1][5] = false;//"An non fiscal receipt is opened to print a 90-degree text"
        myCriticalStatusSet[1][4] = false;//"DialogStorno receipt is  Open"
        myCriticalStatusSet[1][3] = false;//"Low battery (Real time clock is RESET)"
        myCriticalStatusSet[1][2] = false;//"RAM Reset is done."
        myCriticalStatusSet[1][1] = true;//"# Command is not permitted"
        myCriticalStatusSet[1][0] = true;//"# Overflow during command execution"
        myCriticalStatusSet[2][7] = false;//"For internal use  1."
        myCriticalStatusSet[2][6] = false;//"EJ nearly full.(only few receipt allowed)"
        myCriticalStatusSet[2][5] = false;//"Nonfiscal receipt is openFiscalReceipt."
        myCriticalStatusSet[2][4] = false;//"EJ nearly full."
        myCriticalStatusSet[2][3] = false;//"Fiscal receipt is Open."
        myCriticalStatusSet[2][2] = true;//"EJ full."
        myCriticalStatusSet[2][1] = false;//"Near paper end."
        myCriticalStatusSet[2][0] = true;//"#End of paper."
        myCriticalStatusSet[3][7] = false; //"For internal use  1"
        myCriticalStatusSet[3][6] = false; //"Sw7"
        myCriticalStatusSet[3][5] = false; //"Sw6"
        myCriticalStatusSet[3][4] = false; //"Sw5"
        myCriticalStatusSet[3][3] = false; //"Sw4"
        myCriticalStatusSet[3][2] = false; //"Sw3"
        myCriticalStatusSet[3][1] = false; //"Sw2"
        myCriticalStatusSet[3][0] = false; //"Sw1"
        myCriticalStatusSet[4][7] = false;//"For internal use  1"
        myCriticalStatusSet[4][6] = true;//"The printer is overheated"
        myCriticalStatusSet[4][5] = false; //"OR of all errors marked with *  Bytes 4 - 5"
        myCriticalStatusSet[4][4] = true;//"* Fiscal memory is full."
        myCriticalStatusSet[4][3] = false; //"There is space for less then 50 reports in Fiscal memory."
        myCriticalStatusSet[4][2] = false; //"Serial number and number of FM is set"
        myCriticalStatusSet[4][1] = false;//"EIK on BULSTAT is set"
        myCriticalStatusSet[4][0] = true;//"* Error accessing data in the FM"
        myCriticalStatusSet[5][7] = false;//"For internal use  1."
        myCriticalStatusSet[5][6] = false;//"For internal use  0."
        myCriticalStatusSet[5][5] = true; //"Error reading  fiscal memory."
        myCriticalStatusSet[5][4] = false; //"VAT are not set at least once."
        myCriticalStatusSet[5][3] = false;//"Device is not fiscalized."
        myCriticalStatusSet[5][2] = true; //"The latest fiscal memory record failed."
        myCriticalStatusSet[5][1] = true;//"FM is not formated."
        myCriticalStatusSet[5][0] = true;//"The fiscal memory is set in READONLY mode (locked)"
        return myCriticalStatusSet;

    }

    /**
     * Enables whether a status is a critical exception
     *
     * @return
     */
    static boolean[][] initCriticalStatusECR() {

        boolean[][] myCriticalStatusSet = new boolean[6][8];
        myCriticalStatusSet[0][7] = false;//"For internal use  1."
        myCriticalStatusSet[0][6] = false;//"For internal use  0."
        myCriticalStatusSet[0][5] = false;// General error - this is OR of all errors marked with #.
        myCriticalStatusSet[0][4] = false;//""For internal use  0."
        myCriticalStatusSet[0][3] = false;//"No client display connected"
        myCriticalStatusSet[0][2] = false;//"The real time clock is not synchronized"
        myCriticalStatusSet[0][1] = true;//"# Command code is invalid"
        myCriticalStatusSet[0][0] = true;//"# Syntax error"
        myCriticalStatusSet[1][7] = false;//"For internal use  1"
        myCriticalStatusSet[1][6] = true;//"The built-in tax terminal does not responding"
        myCriticalStatusSet[1][5] = false;//"An non fiscal receipt is opened to print a 90-degree text"
        myCriticalStatusSet[1][4] = false;//"receipt is  Open"
        myCriticalStatusSet[1][3] = false;//"For internal use  0."
        myCriticalStatusSet[1][2] = false;//"For internal use  0."
        myCriticalStatusSet[1][1] = true;//"# Command is not permitted"
        myCriticalStatusSet[1][0] = true;//"# Overflow during command execution"
        myCriticalStatusSet[2][7] = false;//"For internal use  1."
        myCriticalStatusSet[2][6] = false;//"EJ nearly full.(only few receipt allowed)"
        myCriticalStatusSet[2][5] = false;//"Nonfiscal receipt is openFiscalReceipt."
        myCriticalStatusSet[2][4] = false;//"EJ nearly full."
        myCriticalStatusSet[2][3] = false;//"Fiscal receipt is Open."
        myCriticalStatusSet[2][2] = false;//"EJ full." // TODO:
        myCriticalStatusSet[2][1] = false;//"Near paper end."
        myCriticalStatusSet[2][0] = true;//"#End of paper."
        myCriticalStatusSet[3][7] = false; //"For internal use  1"
        myCriticalStatusSet[3][6] = false; //"Sw7"
        myCriticalStatusSet[3][5] = false; //"Sw6"
        myCriticalStatusSet[3][4] = false; //"Sw5"
        myCriticalStatusSet[3][3] = false; //"Sw4"
        myCriticalStatusSet[3][2] = false; //"Sw3"
        myCriticalStatusSet[3][1] = false; //"Sw2"
        myCriticalStatusSet[3][0] = false; //"Sw1"
        myCriticalStatusSet[4][7] = false;//"For internal use  1"
        myCriticalStatusSet[4][6] = true;//"The printer is overheated"
        myCriticalStatusSet[4][5] = false; //"OR of all errors marked with *  Bytes 4 - 5"
        myCriticalStatusSet[4][4] = true;//"* Fiscal memory is full."
        myCriticalStatusSet[4][3] = false; //"There is space for less then 50 reports in Fiscal memory."
        myCriticalStatusSet[4][2] = false; //"Serial number and number of FM is set"
        myCriticalStatusSet[4][1] = false;//"EIK on BULSTAT is set"
        myCriticalStatusSet[4][0] = true;//"* Error accessing data in the FM"
        myCriticalStatusSet[4][0] = true; //"* Error accessing data in the FM"
        myCriticalStatusSet[5][7] = false;//"For internal use  1."
        myCriticalStatusSet[5][6] = false;//"For internal use  0."
        myCriticalStatusSet[5][5] = true; //"Error reading  fiscal memory."
        myCriticalStatusSet[5][4] = false; //"VAT are not set at least once."
        myCriticalStatusSet[5][3] = false;//"Device is not fiscalized."
        myCriticalStatusSet[5][2] = true; //"The latest fiscal memory record failed."
        myCriticalStatusSet[5][1] = true;//"FM is not formated."
        myCriticalStatusSet[5][0] = true;//"The fiscal memory is set in READONLY mode (locked)"
        return myCriticalStatusSet;
    }

}
