/**
 * @author DatecsSoft_Team
 */
package connectivity;
import com.datecs.fiscalprinter.SDK.FiscalDeviceV2;
import com.datecs.fiscalprinter.SDK.model.BGR.*;
import com.datecs.fiscalprinter.SDK.model.DatecsFiscalDevice;
import com.fazecast.jSerialComm.SerialPort;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import static com.datecs.fiscalprinter.SDK.FiscalException.FPTR_CC_OTHER;


/**
 * <p>
 */
public class SerialPrinterConnector implements PrinterConnection {
    private SerialPort serialPort = null;

    private static String modelName = "";

    public static DatecsFiscalDevice getConnectedFD() throws Exception {
        if (connectedFD.isConnectedDeviceV2()) return connectedFD;
        else throw new Exception("There is no supported Device connected");
    }

    private static DatecsFiscalDevice connectedFD = new DatecsFiscalDevice(FPTR_CC_OTHER);

    public static String getModelName() {
        return modelName;
    }

    private int boudrate = 115200;
    private int timeOut = 0;

    public SerialPort getConnection() throws Exception {
        return serialPort;
    }

    public void checkOpened() throws Exception {
        if (serialPort == null) {
            throw new Exception("Port is not opened");
        }
    }

    @Override
    public String getPortName() throws Exception {
        return serialPort.getSystemPortName();
    }

    /**
     * <p>
     * By this method we open the serial port and ask the model vendor name, if the model
     * is from the supported in this SDK, we create an instance of the respective model class.
     * <p>
     * The following fiscal devices are supported in this demo:
     * <p>
     * DATECS FMP-350X
     * DATECS FMP-55X
     * DATECS FP-700X
     * DATECS WP-500X
     * DATECS WP-50X
     * DATECS DP-25X
     * DATECS DP-150X
     *
     * @param portName Serial COM Port
     * @return DatecsFiscalDevice
     * @throws Exception
     */
    @Override
    public DatecsFiscalDevice open(String portName) throws Exception {
        connectedFD = new DatecsFiscalDevice(FPTR_CC_OTHER);//Status Bytes description and Exception messages in English language
        try {
            if (serialPort == null) {
                serialPort = SerialPort.getCommPort(portName);
                if (!serialPort.openPort()) throw new Exception("Can't open port :" + portName);

                serialPort.setComPortParameters(boudrate, 8, 1, 0);
                serialPort.setComPortTimeouts(SerialPort.TIMEOUT_READ_SEMI_BLOCKING, timeOut, 0);
                serialPort.setFlowControl(SerialPort.FLOW_CONTROL_DISABLED);
                InputStream in = serialPort.getInputStream();
                OutputStream out = serialPort.getOutputStream();


                /**
                 *  Datecs JavaSDK types of exceptions:
                 *
                 *   The FiscalException are throws in Datecs JavaSDK,
                 *
                 *  FiscalException of Status Bytes they depend on whether a status is marked as critical or not,
                 *  the SDK method setIsStatusCritical() - enables SDK users to select Status of device
                 *  to be a critical exception for their application or not.
                 *  For example, Status of (Byte[0] Bit[3])-"No client display connected"
                 *  may be a critical error in a POS application running in a store but
                 *  if fiscal printer is an office application and no need of client display, this exception must to turned off.
                 *
                 *  FiscalException - Command Error
                 *  These are all exceptions for which a Fiscal Device command can not be executed,
                 *  after the execution of each command, the device returns a response contain error code,
                 *  If the error code is different 0-OK, Datecs JavaSDK - converts the code into messages and added it to
                 *
                 *   Two setters define the content of the message:
                 *   setThrowErrorCode() -about the error code.
                 *   setThrowErrorMessage()-for the description of the error.
                 *
                 * Use fiscalDevice.getConnectedPrinterV2().setMsgSeparator(); to setup desired separator char or string,
                 * by default separator is "\n\r"
                 *
                 * All messages are in English.
                 */

                //Enables whether a status of the device is a critical exception for user application !
                FiscalDeviceV2.setIsStatusCritical(initCriticalStatus());
                //Autodetect model
                FDModelDetectorV2 datecsBGRmodelV2 = new FDModelDetectorV2(
                        serialPort.getInputStream(),
                        serialPort.getOutputStream());
                modelName = datecsBGRmodelV2.detectConnectedModel();

                switch (modelName) {
                    case "FMP-350X":
                        connectedFD.setConnectedModel(new FMP350X_BGR(datecsBGRmodelV2.getTransportProtocol()));
                        return connectedFD;
                    case "FMP-55X":
                        connectedFD.setConnectedModel(new FMP55X_BGR(datecsBGRmodelV2.getTransportProtocol()));
                        return connectedFD;
                    case "FP-700X":
                        connectedFD.setConnectedModel(new FP700X_BGR(datecsBGRmodelV2.getTransportProtocol()));
                        return connectedFD;
                    case "WP-500X":
                        connectedFD.setConnectedModel(new WP500X_BGR(datecsBGRmodelV2.getTransportProtocol()));
                        //datecsBGRmodelV2.getTransportProtocol().setDebugLevel(Level.ALL);
                        return connectedFD;
                    case "WP-50X":
                        connectedFD.setConnectedModel(new WP50X_BGR(datecsBGRmodelV2.getTransportProtocol()));
                        return connectedFD;
                    case "DP-25X":
                        connectedFD.setConnectedModel(new DP25X_BGR(datecsBGRmodelV2.getTransportProtocol()));
                        return connectedFD;
                    case "DP-150X":
                        connectedFD.setConnectedModel(new DP150X_BGR(datecsBGRmodelV2.getTransportProtocol()));
                        return connectedFD;
                    default: {
                        close();
                        throw new Exception("Unsupported model:" + modelName);
                    }
                }

            }

        } catch (Exception ex) {
            close();
            throw new Exception(ex.getMessage());
        }
        return null;
    }

    @Override
    public void close() throws IOException {
        if (isOpen()) {
            serialPort.getInputStream().close();
            serialPort.getOutputStream().close();
            serialPort.closePort();
        }
    }

    private boolean isOpen() {
        if (serialPort == null) {
            return false;
        }
        return serialPort.isOpen();
    }

    @Override
    public boolean isSearchByBaudRateEnabled() {
        return false;
    }

    @Override
    public void setBaudRate(int baudRate) throws Exception {
        if (this.boudrate != baudRate) {
            this.boudrate = baudRate;
        }

    }

    @Override
    public void setTimeout(int timeout) throws Exception {
        this.timeOut = timeout;
    }

    @Override
    public void setPortName(String portName) throws Exception {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    public InputStream getInputStream() {
        return serialPort.getInputStream();
    }

    public OutputStream getOutputStream() {
        return serialPort.getOutputStream();
    }



}
