#import "SerialStream.h"

#define LOG_ENABLED

#ifndef LOG_ENABLED
#undef LOG
#define LOG(...) //
#undef LOGDATA
#define LOGDATA(title,buf,len) //
#endif

@implementation SerialStream

@synthesize connected;


@synthesize controlDelegate;

#pragma mark - Init stuff
- (id)initWithPort:(ORSSerialPort *)port
{
    self = [super init];
    if (self) {
        // Initialization code here.
        streamStatus = NSStreamStatusNotOpen;
        streamData=[[NSMutableData alloc] init];
        lock=[[NSLock alloc] init];
        
        readLen=0;
        
        serial=port;
        serial.delegate=self;
    }
    
    return self;
}

- (void)dealloc
{
}

-(void)connect
{
}
-(void)disconnect
{
}

- (void)clear
{
    if([streamData length])
    {
        [streamData setLength:0];
    }
}

#define GLOBAL_MAX_DEBUG_DATA_LENGTH 512
static NSString *toHexString(const uint8_t *data, size_t length, BOOL ascii)
{
    if (length <= 0) {
        return @"";
    }
    
    const char HEX[]="0123456789ABCDEF";
    int i,j;
    static char s[GLOBAL_MAX_DEBUG_DATA_LENGTH*6+100];
    for(i=0,j=0;i<length && i<GLOBAL_MAX_DEBUG_DATA_LENGTH;i++)
    {
        if(ascii)
        {
            s[j++]='(';
            if(data[i]<0x20)
                s[j++]='.';
            else
                s[j++]=data[i];
            s[j++]=')';
        }
        s[j++]=HEX[data[i]>>4];
        s[j++]=HEX[data[i]&0x0f];
        s[j++]=' ';
    }
    if(length>GLOBAL_MAX_DEBUG_DATA_LENGTH)
    {
        s[j++]='.';
        s[j++]='.';
        s[j++]='.';
    }
    s[j]=0;
    return [NSString stringWithCString:s encoding:NSASCIIStringEncoding];
}


#pragma mark - NSStream subclass overrides

- (void)open
{
    if(streamStatus!=NSStreamStatusOpen && streamStatus!=NSStreamStatusOpening)
    {
        streamStatus=NSStreamStatusOpening;
    
        serial.baudRate=[NSNumber numberWithInt:115200];
        serial.numberOfStopBits=1;
        serial.parity=ORSSerialPortParityNone;
        serial.usesDCDOutputFlowControl=false;
        serial.usesDTRDSRFlowControl=false;
        serial.usesRTSCTSFlowControl=false;
        serial.shouldEchoReceivedData=false;
        
        [serial open];
    }
}

- (void)close
{
    if(streamStatus!=NSStreamStatusClosed)
    {
        [self serialPortWasClosed:serial];
    }
}

- (id<NSStreamDelegate>)delegate
{
    return delegate;
}

- (void)setDelegate:(id<NSStreamDelegate>)aDelegate
{
    delegate = aDelegate;
}

- (void)scheduleInRunLoop:(NSRunLoop *)aRunLoop forMode:(NSString *)mode
{
}

- (void)removeFromRunLoop:(NSRunLoop *)aRunLoop forMode:(NSString *)mode
{
}

- (id)propertyForKey:(NSString *)key
{
    return nil;
}

- (BOOL)setProperty:(id)property forKey:(NSString *)key
{
    return NO;
}

- (NSStreamStatus)streamStatus
{
    return streamStatus;
}

- (NSError *)streamError
{
    return nil;
}


#pragma mark - NSInputStream subclass overrides

- (NSInteger)read:(uint8_t *)buffer maxLength:(NSUInteger)len
{
    [lock lock];
    if(len>[streamData length])
        len=[streamData length];
    
    if(len)
    {
        [streamData getBytes:buffer length:len];
        [streamData replaceBytesInRange:NSMakeRange(0,len) withBytes:nil length:0];
    }
    [lock unlock];
	return len;
}

- (bool)getBuffer:(uint8_t **)buffer length:(NSUInteger *)len
{
	// Not appropriate for this kind of stream; return NO.
    [lock lock];
    *buffer=(uint8_t *)[streamData bytes];
    *len=[streamData length];
    [lock unlock];
	return YES;
}

- (bool)hasBytesAvailable
{
	return [streamData length]>0;
}

#pragma mark - NSOutputStream subclass overrides

- (NSInteger)write:(const uint8_t *)buffer maxLength:(NSUInteger)length
{
    NSLog(@"SW: %@",toHexString(buffer, length, false));
    if([serial sendData:[NSData dataWithBytes:buffer length:length]])
        return length;
    return 0;
}

- (bool)hasSpaceAvailable
{
	return YES;
}

- (NSInteger)write:(NSString *)str
{
    return [self write:(const uint8_t *)[str UTF8String] maxLength:[str length]];
}

#pragma mark - ORSerialPort delegate

/**
 *  Called when new data is received by the serial port from an external source.
 *
 *  @param serialPort The `ORSSerialPort` instance representing the port that received `data`.
 *  @param data       An `NSData` instance containing the data received.
 */
- (void)serialPort:(ORSSerialPort *)serialPort didReceiveData:(NSData *)data
{
    NSLog(@"SR: %@",toHexString(data.bytes, data.length, false));
    [lock lock];
    [streamData appendData:data];
    [lock unlock];
    if([delegate respondsToSelector:@selector(stream:handleEvent:)])
        [delegate stream:self handleEvent:NSStreamEventHasBytesAvailable];
}

/**
 *  Called when a serial port is removed from the system, e.g. the user unplugs
 *  the USB to serial adapter for the port.
 *
 *	In this method, you should discard any strong references you have maintained for the
 *  passed in `serialPort` object. The behavior of `ORSSerialPort` instances whose underlying
 *  serial port has been removed from the system is undefined.
 *
 *  @param serialPort The `ORSSerialPort` instance representing the port that was removed.
 */
- (void)serialPortWasRemovedFromSystem:(ORSSerialPort *)serialPort
{
    NSLog(@"serialPortWasRemovedFromSystem");
    [self serialPortWasClosed:serialPort];
}

/**
 *  Called when an error occurs during an operation involving a serial port.
 *
 *	This method is always used to report errors. No `ORSSerialPort` methods
 *  take a passed in `NSError **` reference because errors may occur asynchonously,
 *  after a method has returned.
 *
 *	Currently, errors reported using this method are always in the `NSPOSIXErrorDomain`,
 *  and a list of error codes can be found in the system header errno.h.
 *
 *  The error object's userInfo dictionary contains the following keys:
 *
 *	- NSLocalizedDescriptionKey - An error message string.
 *	- NSFilePathErrorKey - The device path to the serial port. Same as `[serialPort path]`.
 *
 *  @param serialPort The `ORSSerialPort` instance for the port
 *  @param error      An `NSError` object containing information about the error.
 */
- (void)serialPort:(ORSSerialPort *)serialPort didEncounterError:(NSError *)error
{
    NSLog(@"serialPort:didEncounterError");
    if([delegate respondsToSelector:@selector(stream:handleEvent:)])
        [delegate stream:self handleEvent:NSStreamEventErrorOccurred];
}

/**
 *  Called when a serial port is successfully opened.
 *
 *  @param serialPort The `ORSSerialPort` instance representing the port that was opened.
 */
- (void)serialPortWasOpened:(ORSSerialPort *)serialPort
{
    NSLog(@"serialPortWasOpened");
    streamStatus=NSStreamStatusOpen;
    if([delegate respondsToSelector:@selector(stream:handleEvent:)])
        [delegate stream:self handleEvent:NSStreamEventOpenCompleted];
}

/**
 *  Called when a serial port was closed (e.g. because `-close`) was called.
 *
 *  @param serialPort The `ORSSerialPort` instance representing the port that was closed.
 */
- (void)serialPortWasClosed:(ORSSerialPort *)serialPort
{
    NSLog(@"serialPortWasClosed");
    [serial close];
    streamStatus=NSStreamStatusClosed;
    if([delegate respondsToSelector:@selector(stream:handleEvent:)])
        [delegate stream:self handleEvent:NSStreamEventEndEncountered];
}

@end

